import {
  users,
  categories,
  subcategories,
  menuItems,
  modifierCategories,
  modifiers,
  menuItemModifiers,
  addonCategories,
  addons,
  menuItemAddons,
  feedback,
  settings,
  type User,
  type InsertUser,
  type Category,
  type InsertCategory,
  type Subcategory,
  type InsertSubcategory,
  type MenuItem,
  type InsertMenuItem,
  type ModifierCategory,
  type InsertModifierCategory,
  type Modifier,
  type InsertModifier,
  type AddonCategory,
  type InsertAddonCategory,
  type Addon,
  type InsertAddon,
  type Feedback,
  type InsertFeedback,
  type Setting,
  type InsertSetting,
} from "@shared/schema";
import { db } from "./db";
import { eq, and, desc, asc } from "drizzle-orm";

export interface IStorage {
  // User operations
  getUser(id: number): Promise<User | undefined>;
  getUserByUsername(username: string): Promise<User | undefined>;
  createUser(user: InsertUser): Promise<User>;
  updateUser(id: number, user: Partial<InsertUser>): Promise<User>;
  deleteUser(id: number): Promise<void>;
  getAllUsers(): Promise<User[]>;

  // Category operations
  getAllCategories(): Promise<Category[]>;
  getActiveCategories(): Promise<Category[]>;
  getCategory(id: number): Promise<Category | undefined>;
  createCategory(category: InsertCategory): Promise<Category>;
  updateCategory(id: number, category: Partial<InsertCategory>): Promise<Category>;
  deleteCategory(id: number): Promise<void>;

  // Subcategory operations
  getAllSubcategories(): Promise<Subcategory[]>;
  getSubcategoriesByCategory(categoryId: number): Promise<Subcategory[]>;
  createSubcategory(subcategory: InsertSubcategory): Promise<Subcategory>;
  updateSubcategory(id: number, subcategory: Partial<InsertSubcategory>): Promise<Subcategory>;
  deleteSubcategory(id: number): Promise<void>;

  // Menu item operations
  getAllMenuItems(): Promise<MenuItem[]>;
  getActiveMenuItems(): Promise<MenuItem[]>;
  getMenuItemsByCategory(categoryId: number): Promise<MenuItem[]>;
  getMenuItem(id: number): Promise<MenuItem | undefined>;
  createMenuItem(item: InsertMenuItem): Promise<MenuItem>;
  updateMenuItem(id: number, item: Partial<InsertMenuItem>): Promise<MenuItem>;
  deleteMenuItem(id: number): Promise<void>;

  // Modifier operations
  getAllModifierCategories(): Promise<ModifierCategory[]>;
  getModifiersByCategory(categoryId: number): Promise<Modifier[]>;
  createModifierCategory(category: InsertModifierCategory): Promise<ModifierCategory>;
  createModifier(modifier: InsertModifier): Promise<Modifier>;
  updateModifier(id: number, modifier: Partial<InsertModifier>): Promise<Modifier>;
  deleteModifier(id: number): Promise<void>;

  // Addon operations
  getAllAddonCategories(): Promise<AddonCategory[]>;
  getAddonsByCategory(categoryId: number): Promise<Addon[]>;
  createAddonCategory(category: InsertAddonCategory): Promise<AddonCategory>;
  createAddon(addon: InsertAddon): Promise<Addon>;
  updateAddon(id: number, addon: Partial<InsertAddon>): Promise<Addon>;
  deleteAddon(id: number): Promise<void>;

  // Feedback operations
  getAllFeedback(): Promise<Feedback[]>;
  getFeedback(id: number): Promise<Feedback | undefined>;
  createFeedback(feedbackData: InsertFeedback): Promise<Feedback>;
  archiveFeedback(id: number): Promise<void>;

  // Settings operations
  getSetting(key: string): Promise<Setting | undefined>;
  setSetting(setting: InsertSetting): Promise<Setting>;
  updateSetting(key: string, value: any): Promise<Setting>;
}

export class DatabaseStorage implements IStorage {
  // User operations
  async getUser(id: number): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.id, id));
    return user || undefined;
  }

  async getUserByUsername(username: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.username, username));
    return user || undefined;
  }

  async createUser(userData: InsertUser): Promise<User> {
    const [user] = await db.insert(users).values(userData).returning();
    return user;
  }

  async updateUser(id: number, userData: Partial<InsertUser>): Promise<User> {
    const [user] = await db
      .update(users)
      .set({ ...userData, updatedAt: new Date() })
      .where(eq(users.id, id))
      .returning();
    return user;
  }

  async deleteUser(id: number): Promise<void> {
    await db.delete(users).where(eq(users.id, id));
  }

  async getAllUsers(): Promise<User[]> {
    return await db.select().from(users).orderBy(asc(users.username));
  }

  // Category operations
  async getAllCategories(): Promise<Category[]> {
    return await db.select().from(categories).orderBy(asc(categories.sortOrder), asc(categories.name));
  }

  async getActiveCategories(): Promise<Category[]> {
    return await db.select().from(categories)
      .where(eq(categories.isActive, true))
      .orderBy(asc(categories.sortOrder), asc(categories.name));
  }

  async getCategory(id: number): Promise<Category | undefined> {
    const [category] = await db.select().from(categories).where(eq(categories.id, id));
    return category || undefined;
  }

  async createCategory(categoryData: InsertCategory): Promise<Category> {
    const [category] = await db.insert(categories).values(categoryData).returning();
    return category;
  }

  async updateCategory(id: number, categoryData: Partial<InsertCategory>): Promise<Category> {
    const [category] = await db
      .update(categories)
      .set({ ...categoryData, updatedAt: new Date() })
      .where(eq(categories.id, id))
      .returning();
    return category;
  }

  async deleteCategory(id: number): Promise<void> {
    await db.delete(categories).where(eq(categories.id, id));
  }

  // Subcategory operations
  async getAllSubcategories(): Promise<Subcategory[]> {
    return await db.select().from(subcategories).orderBy(asc(subcategories.sortOrder), asc(subcategories.name));
  }

  async getSubcategoriesByCategory(categoryId: number): Promise<Subcategory[]> {
    return await db.select().from(subcategories)
      .where(and(eq(subcategories.categoryId, categoryId), eq(subcategories.isActive, true)))
      .orderBy(asc(subcategories.sortOrder), asc(subcategories.name));
  }

  async createSubcategory(subcategoryData: InsertSubcategory): Promise<Subcategory> {
    const [subcategory] = await db.insert(subcategories).values(subcategoryData).returning();
    return subcategory;
  }

  async updateSubcategory(id: number, subcategoryData: Partial<InsertSubcategory>): Promise<Subcategory> {
    const [subcategory] = await db
      .update(subcategories)
      .set({ ...subcategoryData, updatedAt: new Date() })
      .where(eq(subcategories.id, id))
      .returning();
    return subcategory;
  }

  async deleteSubcategory(id: number): Promise<void> {
    await db.delete(subcategories).where(eq(subcategories.id, id));
  }

  // Menu item operations
  async getAllMenuItems(): Promise<MenuItem[]> {
    return await db.select().from(menuItems).orderBy(asc(menuItems.sortOrder), asc(menuItems.name));
  }

  async getActiveMenuItems(): Promise<MenuItem[]> {
    return await db.select().from(menuItems)
      .where(eq(menuItems.isActive, true))
      .orderBy(asc(menuItems.sortOrder), asc(menuItems.name));
  }

  async getMenuItemsByCategory(categoryId: number): Promise<MenuItem[]> {
    return await db.select().from(menuItems)
      .where(and(eq(menuItems.categoryId, categoryId), eq(menuItems.isActive, true)))
      .orderBy(asc(menuItems.sortOrder), asc(menuItems.name));
  }

  async getMenuItem(id: number): Promise<MenuItem | undefined> {
    const [item] = await db.select().from(menuItems).where(eq(menuItems.id, id));
    return item || undefined;
  }

  async createMenuItem(itemData: InsertMenuItem): Promise<MenuItem> {
    const [item] = await db.insert(menuItems).values(itemData).returning();
    return item;
  }

  async updateMenuItem(id: number, itemData: Partial<InsertMenuItem>): Promise<MenuItem> {
    const [item] = await db
      .update(menuItems)
      .set({ ...itemData, updatedAt: new Date() })
      .where(eq(menuItems.id, id))
      .returning();
    return item;
  }

  async deleteMenuItem(id: number): Promise<void> {
    await db.delete(menuItems).where(eq(menuItems.id, id));
  }

  // Modifier operations
  async getAllModifierCategories(): Promise<ModifierCategory[]> {
    return await db.select().from(modifierCategories).orderBy(asc(modifierCategories.name));
  }

  async getModifiersByCategory(categoryId: number): Promise<Modifier[]> {
    return await db.select().from(modifiers)
      .where(and(eq(modifiers.categoryId, categoryId), eq(modifiers.isActive, true)))
      .orderBy(asc(modifiers.sortOrder), asc(modifiers.name));
  }

  async createModifierCategory(categoryData: InsertModifierCategory): Promise<ModifierCategory> {
    const [category] = await db.insert(modifierCategories).values(categoryData).returning();
    return category;
  }

  async createModifier(modifierData: InsertModifier): Promise<Modifier> {
    const [modifier] = await db.insert(modifiers).values(modifierData).returning();
    return modifier;
  }

  async updateModifier(id: number, modifierData: Partial<InsertModifier>): Promise<Modifier> {
    const [modifier] = await db
      .update(modifiers)
      .set({ ...modifierData, updatedAt: new Date() })
      .where(eq(modifiers.id, id))
      .returning();
    return modifier;
  }

  async deleteModifier(id: number): Promise<void> {
    await db.delete(modifiers).where(eq(modifiers.id, id));
  }

  // Addon operations
  async getAllAddonCategories(): Promise<AddonCategory[]> {
    return await db.select().from(addonCategories).orderBy(asc(addonCategories.name));
  }

  async getAddonsByCategory(categoryId: number): Promise<Addon[]> {
    return await db.select().from(addons)
      .where(and(eq(addons.categoryId, categoryId), eq(addons.isActive, true)))
      .orderBy(asc(addons.sortOrder), asc(addons.name));
  }

  async createAddonCategory(categoryData: InsertAddonCategory): Promise<AddonCategory> {
    const [category] = await db.insert(addonCategories).values(categoryData).returning();
    return category;
  }

  async createAddon(addonData: InsertAddon): Promise<Addon> {
    const [addon] = await db.insert(addons).values(addonData).returning();
    return addon;
  }

  async updateAddon(id: number, addonData: Partial<InsertAddon>): Promise<Addon> {
    const [addon] = await db
      .update(addons)
      .set({ ...addonData, updatedAt: new Date() })
      .where(eq(addons.id, id))
      .returning();
    return addon;
  }

  async deleteAddon(id: number): Promise<void> {
    await db.delete(addons).where(eq(addons.id, id));
  }

  // Feedback operations
  async getAllFeedback(): Promise<Feedback[]> {
    return await db.select().from(feedback)
      .where(eq(feedback.isArchived, false))
      .orderBy(desc(feedback.createdAt));
  }

  async getFeedback(id: number): Promise<Feedback | undefined> {
    const [feedbackItem] = await db.select().from(feedback).where(eq(feedback.id, id));
    return feedbackItem || undefined;
  }

  async createFeedback(feedbackData: InsertFeedback): Promise<Feedback> {
    const [feedbackItem] = await db.insert(feedback).values(feedbackData).returning();
    return feedbackItem;
  }

  async archiveFeedback(id: number): Promise<void> {
    await db
      .update(feedback)
      .set({ isArchived: true, updatedAt: new Date() })
      .where(eq(feedback.id, id));
  }

  // Settings operations
  async getSetting(key: string): Promise<Setting | undefined> {
    const [setting] = await db.select().from(settings).where(eq(settings.key, key));
    return setting || undefined;
  }

  async setSetting(settingData: InsertSetting): Promise<Setting> {
    const [setting] = await db.insert(settings).values(settingData).returning();
    return setting;
  }

  async updateSetting(key: string, value: any): Promise<Setting> {
    const [setting] = await db
      .update(settings)
      .set({ value, updatedAt: new Date() })
      .where(eq(settings.key, key))
      .returning();
    return setting;
  }
}

export const storage = new DatabaseStorage();
