import type { Express } from "express";
import { createServer, type Server } from "http";
import { storage } from "./storage";
import { insertUserSchema, insertCategorySchema, insertMenuItemSchema, insertFeedbackSchema } from "@shared/schema";
import bcrypt from "bcrypt";
import jwt from "jsonwebtoken";

const JWT_SECRET = process.env.JWT_SECRET || "your-secret-key";

// Middleware to verify JWT token
const authenticateToken = (req: any, res: any, next: any) => {
  const authHeader = req.headers['authorization'];
  const token = authHeader && authHeader.split(' ')[1];

  if (!token) {
    return res.status(401).json({ message: 'Access token required' });
  }

  jwt.verify(token, JWT_SECRET, (err: any, user: any) => {
    if (err) {
      return res.status(403).json({ message: 'Invalid or expired token' });
    }
    req.user = user;
    next();
  });
};

export async function registerRoutes(app: Express): Promise<Server> {
  
  // Authentication routes
  app.post('/api/auth/login', async (req, res) => {
    try {
      const { username, password } = req.body;
      
      if (!username || !password) {
        return res.status(400).json({ message: 'Username and password required' });
      }

      const user = await storage.getUserByUsername(username);
      if (!user || !user.isActive) {
        return res.status(401).json({ message: 'Invalid credentials' });
      }

      const validPassword = await bcrypt.compare(password, user.password);
      if (!validPassword) {
        return res.status(401).json({ message: 'Invalid credentials' });
      }

      const token = jwt.sign(
        { id: user.id, username: user.username, role: user.role },
        JWT_SECRET,
        { expiresIn: '24h' }
      );

      res.json({
        token,
        user: {
          id: user.id,
          username: user.username,
          email: user.email,
          firstName: user.firstName,
          lastName: user.lastName,
          role: user.role
        }
      });
    } catch (error) {
      console.error('Login error:', error);
      res.status(500).json({ message: 'Login failed' });
    }
  });

  app.get('/api/auth/verify', authenticateToken, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.id);
      if (!user || !user.isActive) {
        return res.status(401).json({ message: 'User not found or inactive' });
      }

      res.json({
        user: {
          id: user.id,
          username: user.username,
          email: user.email,
          firstName: user.firstName,
          lastName: user.lastName,
          role: user.role
        }
      });
    } catch (error) {
      console.error('Verify error:', error);
      res.status(500).json({ message: 'Verification failed' });
    }
  });

  // Category routes
  app.get('/api/categories', async (req, res) => {
    try {
      const categories = await storage.getActiveCategories();
      
      // Filter categories based on time availability
      const now = new Date();
      const currentTime = `${now.getHours().toString().padStart(2, '0')}:${now.getMinutes().toString().padStart(2, '0')}`;
      
      const availableCategories = categories.filter(category => {
        if (!category.startTime || !category.endTime) return true;
        
        const start = category.startTime;
        const end = category.endTime;
        
        // Handle cases where end time is next day (e.g., 22:00 - 06:00)
        if (start <= end) {
          return currentTime >= start && currentTime <= end;
        } else {
          return currentTime >= start || currentTime <= end;
        }
      });

      res.json(availableCategories);
    } catch (error) {
      console.error('Get categories error:', error);
      res.status(500).json({ message: 'Failed to fetch categories' });
    }
  });

  app.get('/api/categories/all', authenticateToken, async (req, res) => {
    try {
      const categories = await storage.getAllCategories();
      res.json(categories);
    } catch (error) {
      console.error('Get all categories error:', error);
      res.status(500).json({ message: 'Failed to fetch categories' });
    }
  });

  app.post('/api/categories', authenticateToken, async (req, res) => {
    try {
      const categoryData = insertCategorySchema.parse(req.body);
      const category = await storage.createCategory(categoryData);
      res.status(201).json(category);
    } catch (error) {
      console.error('Create category error:', error);
      res.status(500).json({ message: 'Failed to create category' });
    }
  });

  app.put('/api/categories/:id', authenticateToken, async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const categoryData = insertCategorySchema.partial().parse(req.body);
      const category = await storage.updateCategory(id, categoryData);
      res.json(category);
    } catch (error) {
      console.error('Update category error:', error);
      res.status(500).json({ message: 'Failed to update category' });
    }
  });

  app.delete('/api/categories/:id', authenticateToken, async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      await storage.deleteCategory(id);
      res.status(204).send();
    } catch (error) {
      console.error('Delete category error:', error);
      res.status(500).json({ message: 'Failed to delete category' });
    }
  });

  // Subcategory routes
  app.get('/api/subcategories/:categoryId', async (req, res) => {
    try {
      const categoryId = parseInt(req.params.categoryId);
      const subcategories = await storage.getSubcategoriesByCategory(categoryId);
      res.json(subcategories);
    } catch (error) {
      console.error('Get subcategories error:', error);
      res.status(500).json({ message: 'Failed to fetch subcategories' });
    }
  });

  // Menu item routes
  app.get('/api/menu-items', async (req, res) => {
    try {
      const { categoryId } = req.query;
      let items;
      
      if (categoryId) {
        items = await storage.getMenuItemsByCategory(parseInt(categoryId as string));
      } else {
        items = await storage.getActiveMenuItems();
      }
      
      res.json(items);
    } catch (error) {
      console.error('Get menu items error:', error);
      res.status(500).json({ message: 'Failed to fetch menu items' });
    }
  });

  app.get('/api/menu-items/all', authenticateToken, async (req, res) => {
    try {
      const items = await storage.getAllMenuItems();
      res.json(items);
    } catch (error) {
      console.error('Get all menu items error:', error);
      res.status(500).json({ message: 'Failed to fetch menu items' });
    }
  });

  app.get('/api/menu-items/:id', async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const item = await storage.getMenuItem(id);
      if (!item) {
        return res.status(404).json({ message: 'Menu item not found' });
      }
      res.json(item);
    } catch (error) {
      console.error('Get menu item error:', error);
      res.status(500).json({ message: 'Failed to fetch menu item' });
    }
  });

  app.post('/api/menu-items', authenticateToken, async (req, res) => {
    try {
      const itemData = insertMenuItemSchema.parse(req.body);
      const item = await storage.createMenuItem(itemData);
      res.status(201).json(item);
    } catch (error) {
      console.error('Create menu item error:', error);
      res.status(500).json({ message: 'Failed to create menu item' });
    }
  });

  app.put('/api/menu-items/:id', authenticateToken, async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const itemData = insertMenuItemSchema.partial().parse(req.body);
      const item = await storage.updateMenuItem(id, itemData);
      res.json(item);
    } catch (error) {
      console.error('Update menu item error:', error);
      res.status(500).json({ message: 'Failed to update menu item' });
    }
  });

  app.delete('/api/menu-items/:id', authenticateToken, async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      await storage.deleteMenuItem(id);
      res.status(204).send();
    } catch (error) {
      console.error('Delete menu item error:', error);
      res.status(500).json({ message: 'Failed to delete menu item' });
    }
  });

  // Modifier routes
  app.get('/api/modifier-categories', authenticateToken, async (req, res) => {
    try {
      const categories = await storage.getAllModifierCategories();
      res.json(categories);
    } catch (error) {
      console.error('Get modifier categories error:', error);
      res.status(500).json({ message: 'Failed to fetch modifier categories' });
    }
  });

  app.get('/api/modifiers/:categoryId', async (req, res) => {
    try {
      const categoryId = parseInt(req.params.categoryId);
      const modifiers = await storage.getModifiersByCategory(categoryId);
      res.json(modifiers);
    } catch (error) {
      console.error('Get modifiers error:', error);
      res.status(500).json({ message: 'Failed to fetch modifiers' });
    }
  });

  // Addon routes
  app.get('/api/addon-categories', authenticateToken, async (req, res) => {
    try {
      const categories = await storage.getAllAddonCategories();
      res.json(categories);
    } catch (error) {
      console.error('Get addon categories error:', error);
      res.status(500).json({ message: 'Failed to fetch addon categories' });
    }
  });

  app.get('/api/addons/:categoryId', async (req, res) => {
    try {
      const categoryId = parseInt(req.params.categoryId);
      const addons = await storage.getAddonsByCategory(categoryId);
      res.json(addons);
    } catch (error) {
      console.error('Get addons error:', error);
      res.status(500).json({ message: 'Failed to fetch addons' });
    }
  });

  // Feedback routes
  app.post('/api/feedback', async (req, res) => {
    try {
      const feedbackData = insertFeedbackSchema.parse(req.body);
      const feedback = await storage.createFeedback(feedbackData);
      res.status(201).json(feedback);
    } catch (error) {
      console.error('Create feedback error:', error);
      res.status(500).json({ message: 'Failed to submit feedback' });
    }
  });

  app.get('/api/feedback', authenticateToken, async (req, res) => {
    try {
      const feedbackList = await storage.getAllFeedback();
      res.json(feedbackList);
    } catch (error) {
      console.error('Get feedback error:', error);
      res.status(500).json({ message: 'Failed to fetch feedback' });
    }
  });

  app.put('/api/feedback/:id/archive', authenticateToken, async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      await storage.archiveFeedback(id);
      res.status(200).json({ message: 'Feedback archived successfully' });
    } catch (error) {
      console.error('Archive feedback error:', error);
      res.status(500).json({ message: 'Failed to archive feedback' });
    }
  });

  // User management routes
  app.get('/api/users', authenticateToken, async (req, res) => {
    try {
      const users = await storage.getAllUsers();
      // Remove password from response
      const safeUsers = users.map(({ password, ...user }) => user);
      res.json(safeUsers);
    } catch (error) {
      console.error('Get users error:', error);
      res.status(500).json({ message: 'Failed to fetch users' });
    }
  });

  app.post('/api/users', authenticateToken, async (req, res) => {
    try {
      const userData = insertUserSchema.parse(req.body);
      // Hash password before storing
      const hashedPassword = await bcrypt.hash(userData.password, 10);
      const userWithHashedPassword = { ...userData, password: hashedPassword };
      
      const user = await storage.createUser(userWithHashedPassword);
      // Remove password from response
      const { password, ...safeUser } = user;
      res.status(201).json(safeUser);
    } catch (error) {
      console.error('Create user error:', error);
      res.status(500).json({ message: 'Failed to create user' });
    }
  });

  // Settings routes
  app.get('/api/settings/:key', async (req, res) => {
    try {
      const { key } = req.params;
      const setting = await storage.getSetting(key);
      if (!setting) {
        return res.status(404).json({ message: 'Setting not found' });
      }
      res.json(setting);
    } catch (error) {
      console.error('Get setting error:', error);
      res.status(500).json({ message: 'Failed to fetch setting' });
    }
  });

  app.put('/api/settings/:key', authenticateToken, async (req, res) => {
    try {
      const { key } = req.params;
      const { value } = req.body;
      
      const existingSetting = await storage.getSetting(key);
      if (existingSetting) {
        const setting = await storage.updateSetting(key, value);
        res.json(setting);
      } else {
        const setting = await storage.setSetting({ key, value });
        res.status(201).json(setting);
      }
    } catch (error) {
      console.error('Update setting error:', error);
      res.status(500).json({ message: 'Failed to update setting' });
    }
  });

  // Dashboard stats
  app.get('/api/stats', authenticateToken, async (req, res) => {
    try {
      const [menuItems, categories, feedbackList, users] = await Promise.all([
        storage.getAllMenuItems(),
        storage.getAllCategories(),
        storage.getAllFeedback(),
        storage.getAllUsers()
      ]);

      const stats = {
        totalItems: menuItems.length,
        activeCategories: categories.filter(c => c.isActive).length,
        feedbackCount: feedbackList.length,
        activeUsers: users.filter(u => u.isActive).length
      };

      res.json(stats);
    } catch (error) {
      console.error('Get stats error:', error);
      res.status(500).json({ message: 'Failed to fetch stats' });
    }
  });

  const httpServer = createServer(app);
  return httpServer;
}
