const express = require('express');
const path = require('path');
const bcrypt = require('bcrypt');
const jwt = require('jsonwebtoken');
const { Pool } = require('@neondatabase/serverless');
const { drizzle } = require('drizzle-orm/neon-serverless');
const { eq, and, desc, asc } = require('drizzle-orm');
const { z } = require('zod');

// Database connection
if (!process.env.DATABASE_URL) {
  throw new Error('DATABASE_URL must be set in environment variables');
}

const pool = new Pool({ connectionString: process.env.DATABASE_URL });
const db = drizzle({ client: pool });

const JWT_SECRET = process.env.JWT_SECRET || 'your-secret-key';

// Express app setup
const app = express();
app.use(express.json());
app.use(express.urlencoded({ extended: false }));

// Serve static files from dist directory
app.use(express.static(path.join(__dirname, 'dist')));

// Authentication middleware
const authenticateToken = (req, res, next) => {
  const authHeader = req.headers['authorization'];
  const token = authHeader && authHeader.split(' ')[1];

  if (!token) {
    return res.status(401).json({ message: 'Access token required' });
  }

  jwt.verify(token, JWT_SECRET, (err, user) => {
    if (err) {
      return res.status(403).json({ message: 'Invalid or expired token' });
    }
    req.user = user;
    next();
  });
};

// Database operations
const storage = {
  async getUserByUsername(username) {
    const result = await pool.query('SELECT * FROM users WHERE username = $1', [username]);
    return result.rows[0];
  },

  async getUser(id) {
    const result = await pool.query('SELECT * FROM users WHERE id = $1', [id]);
    return result.rows[0];
  },

  async getActiveCategories() {
    const result = await pool.query(
      'SELECT * FROM categories WHERE is_active = true ORDER BY sort_order, name'
    );
    return result.rows;
  },

  async getAllCategories() {
    const result = await pool.query('SELECT * FROM categories ORDER BY sort_order, name');
    return result.rows;
  },

  async getActiveMenuItems() {
    const result = await pool.query(
      'SELECT * FROM menu_items WHERE is_active = true ORDER BY sort_order, name'
    );
    return result.rows;
  },

  async getAllMenuItems() {
    const result = await pool.query('SELECT * FROM menu_items ORDER BY sort_order, name');
    return result.rows;
  },

  async getMenuItemsByCategory(categoryId) {
    const result = await pool.query(
      'SELECT * FROM menu_items WHERE category_id = $1 AND is_active = true ORDER BY sort_order, name',
      [categoryId]
    );
    return result.rows;
  },

  async getMenuItem(id) {
    const result = await pool.query('SELECT * FROM menu_items WHERE id = $1', [id]);
    return result.rows[0];
  },

  async createMenuItem(itemData) {
    const { name, nameArabic, description, descriptionArabic, price, categoryId, subcategoryId, image, video, isActive, sortOrder } = itemData;
    const result = await pool.query(
      `INSERT INTO menu_items (name, name_arabic, description, description_arabic, price, category_id, subcategory_id, image, video, is_active, sort_order) 
       VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11) RETURNING *`,
      [name, nameArabic, description, descriptionArabic, price, categoryId, subcategoryId, image, video, isActive, sortOrder]
    );
    return result.rows[0];
  },

  async updateMenuItem(id, itemData) {
    const fields = [];
    const values = [];
    let paramCount = 1;

    Object.entries(itemData).forEach(([key, value]) => {
      if (value !== undefined) {
        const dbKey = key === 'nameArabic' ? 'name_arabic' : 
                     key === 'descriptionArabic' ? 'description_arabic' :
                     key === 'categoryId' ? 'category_id' :
                     key === 'subcategoryId' ? 'subcategory_id' :
                     key === 'isActive' ? 'is_active' :
                     key === 'sortOrder' ? 'sort_order' : key;
        fields.push(`${dbKey} = $${paramCount}`);
        values.push(value);
        paramCount++;
      }
    });

    if (fields.length === 0) return null;

    fields.push('updated_at = NOW()');
    values.push(id);

    const query = `UPDATE menu_items SET ${fields.join(', ')} WHERE id = $${paramCount} RETURNING *`;
    const result = await pool.query(query, values);
    return result.rows[0];
  },

  async deleteMenuItem(id) {
    await pool.query('DELETE FROM menu_items WHERE id = $1', [id]);
  },

  async createCategory(categoryData) {
    const { name, nameArabic, description, descriptionArabic, startTime, endTime, isActive, sortOrder } = categoryData;
    const result = await pool.query(
      `INSERT INTO categories (name, name_arabic, description, description_arabic, start_time, end_time, is_active, sort_order) 
       VALUES ($1, $2, $3, $4, $5, $6, $7, $8) RETURNING *`,
      [name, nameArabic, description, descriptionArabic, startTime, endTime, isActive, sortOrder]
    );
    return result.rows[0];
  },

  async getAllFeedback() {
    const result = await pool.query(
      'SELECT * FROM feedback WHERE is_archived = false ORDER BY created_at DESC'
    );
    return result.rows;
  },

  async createFeedback(feedbackData) {
    const { name, phone, email, message, rating } = feedbackData;
    const result = await pool.query(
      'INSERT INTO feedback (name, phone, email, message, rating) VALUES ($1, $2, $3, $4, $5) RETURNING *',
      [name, phone, email, message, rating || 5]
    );
    return result.rows[0];
  },

  async archiveFeedback(id) {
    await pool.query('UPDATE feedback SET is_archived = true, updated_at = NOW() WHERE id = $1', [id]);
  },

  async getAllUsers() {
    const result = await pool.query('SELECT id, username, email, first_name, last_name, role, is_active, created_at, updated_at FROM users ORDER BY username');
    return result.rows;
  },

  async createUser(userData) {
    const { username, email, password, role, firstName, lastName } = userData;
    const result = await pool.query(
      'INSERT INTO users (username, email, password, role, first_name, last_name) VALUES ($1, $2, $3, $4, $5, $6) RETURNING id, username, email, first_name, last_name, role, is_active, created_at, updated_at',
      [username, email, password, role || 'admin', firstName, lastName]
    );
    return result.rows[0];
  },

  async getSetting(key) {
    const result = await pool.query('SELECT * FROM settings WHERE key = $1', [key]);
    return result.rows[0];
  },

  async updateSetting(key, value) {
    const result = await pool.query(
      'UPDATE settings SET value = $1, updated_at = NOW() WHERE key = $2 RETURNING *',
      [JSON.stringify(value), key]
    );
    return result.rows[0];
  },

  async setSetting(settingData) {
    const { key, value } = settingData;
    const result = await pool.query(
      'INSERT INTO settings (key, value) VALUES ($1, $2) RETURNING *',
      [key, JSON.stringify(value)]
    );
    return result.rows[0];
  }
};

// API Routes

// Authentication routes
app.post('/api/auth/login', async (req, res) => {
  try {
    const { username, password } = req.body;
    
    if (!username || !password) {
      return res.status(400).json({ message: 'Username and password required' });
    }

    const user = await storage.getUserByUsername(username);
    if (!user || !user.is_active) {
      return res.status(401).json({ message: 'Invalid credentials' });
    }

    const validPassword = await bcrypt.compare(password, user.password);
    if (!validPassword) {
      return res.status(401).json({ message: 'Invalid credentials' });
    }

    const token = jwt.sign(
      { id: user.id, username: user.username, role: user.role },
      JWT_SECRET,
      { expiresIn: '24h' }
    );

    res.json({
      token,
      user: {
        id: user.id,
        username: user.username,
        email: user.email,
        firstName: user.first_name,
        lastName: user.last_name,
        role: user.role
      }
    });
  } catch (error) {
    console.error('Login error:', error);
    res.status(500).json({ message: 'Login failed' });
  }
});

app.get('/api/auth/verify', authenticateToken, async (req, res) => {
  try {
    const user = await storage.getUser(req.user.id);
    if (!user || !user.is_active) {
      return res.status(401).json({ message: 'User not found or inactive' });
    }

    res.json({
      user: {
        id: user.id,
        username: user.username,
        email: user.email,
        firstName: user.first_name,
        lastName: user.last_name,
        role: user.role
      }
    });
  } catch (error) {
    console.error('Verify error:', error);
    res.status(500).json({ message: 'Verification failed' });
  }
});

// Category routes
app.get('/api/categories', async (req, res) => {
  try {
    const categories = await storage.getActiveCategories();
    
    // Filter categories based on time availability
    const now = new Date();
    const currentTime = `${now.getHours().toString().padStart(2, '0')}:${now.getMinutes().toString().padStart(2, '0')}`;
    
    const availableCategories = categories.filter(category => {
      if (!category.start_time || !category.end_time) return true;
      
      const start = category.start_time;
      const end = category.end_time;
      
      // Handle cases where end time is next day (e.g., 22:00 - 06:00)
      if (start <= end) {
        return currentTime >= start && currentTime <= end;
      } else {
        return currentTime >= start || currentTime <= end;
      }
    });

    res.json(availableCategories);
  } catch (error) {
    console.error('Get categories error:', error);
    res.status(500).json({ message: 'Failed to fetch categories' });
  }
});

app.get('/api/categories/all', authenticateToken, async (req, res) => {
  try {
    const categories = await storage.getAllCategories();
    res.json(categories);
  } catch (error) {
    console.error('Get all categories error:', error);
    res.status(500).json({ message: 'Failed to fetch categories' });
  }
});

app.post('/api/categories', authenticateToken, async (req, res) => {
  try {
    const category = await storage.createCategory(req.body);
    res.status(201).json(category);
  } catch (error) {
    console.error('Create category error:', error);
    res.status(500).json({ message: 'Failed to create category' });
  }
});

// Menu item routes
app.get('/api/menu-items', async (req, res) => {
  try {
    const { categoryId } = req.query;
    let items;
    
    if (categoryId) {
      items = await storage.getMenuItemsByCategory(parseInt(categoryId));
    } else {
      items = await storage.getActiveMenuItems();
    }
    
    res.json(items);
  } catch (error) {
    console.error('Get menu items error:', error);
    res.status(500).json({ message: 'Failed to fetch menu items' });
  }
});

app.get('/api/menu-items/all', authenticateToken, async (req, res) => {
  try {
    const items = await storage.getAllMenuItems();
    res.json(items);
  } catch (error) {
    console.error('Get all menu items error:', error);
    res.status(500).json({ message: 'Failed to fetch menu items' });
  }
});

app.get('/api/menu-items/:id', async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    const item = await storage.getMenuItem(id);
    if (!item) {
      return res.status(404).json({ message: 'Menu item not found' });
    }
    res.json(item);
  } catch (error) {
    console.error('Get menu item error:', error);
    res.status(500).json({ message: 'Failed to fetch menu item' });
  }
});

app.post('/api/menu-items', authenticateToken, async (req, res) => {
  try {
    const item = await storage.createMenuItem(req.body);
    res.status(201).json(item);
  } catch (error) {
    console.error('Create menu item error:', error);
    res.status(500).json({ message: 'Failed to create menu item' });
  }
});

app.put('/api/menu-items/:id', authenticateToken, async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    const item = await storage.updateMenuItem(id, req.body);
    res.json(item);
  } catch (error) {
    console.error('Update menu item error:', error);
    res.status(500).json({ message: 'Failed to update menu item' });
  }
});

app.delete('/api/menu-items/:id', authenticateToken, async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    await storage.deleteMenuItem(id);
    res.status(204).send();
  } catch (error) {
    console.error('Delete menu item error:', error);
    res.status(500).json({ message: 'Failed to delete menu item' });
  }
});

// Feedback routes
app.post('/api/feedback', async (req, res) => {
  try {
    const feedback = await storage.createFeedback(req.body);
    res.status(201).json(feedback);
  } catch (error) {
    console.error('Create feedback error:', error);
    res.status(500).json({ message: 'Failed to submit feedback' });
  }
});

app.get('/api/feedback', authenticateToken, async (req, res) => {
  try {
    const feedbackList = await storage.getAllFeedback();
    res.json(feedbackList);
  } catch (error) {
    console.error('Get feedback error:', error);
    res.status(500).json({ message: 'Failed to fetch feedback' });
  }
});

app.put('/api/feedback/:id/archive', authenticateToken, async (req, res) => {
  try {
    const id = parseInt(req.params.id);
    await storage.archiveFeedback(id);
    res.status(200).json({ message: 'Feedback archived successfully' });
  } catch (error) {
    console.error('Archive feedback error:', error);
    res.status(500).json({ message: 'Failed to archive feedback' });
  }
});

// User management routes
app.get('/api/users', authenticateToken, async (req, res) => {
  try {
    const users = await storage.getAllUsers();
    res.json(users);
  } catch (error) {
    console.error('Get users error:', error);
    res.status(500).json({ message: 'Failed to fetch users' });
  }
});

app.post('/api/users', authenticateToken, async (req, res) => {
  try {
    const userData = req.body;
    // Hash password before storing
    const hashedPassword = await bcrypt.hash(userData.password, 10);
    const userWithHashedPassword = { ...userData, password: hashedPassword };
    
    const user = await storage.createUser(userWithHashedPassword);
    res.status(201).json(user);
  } catch (error) {
    console.error('Create user error:', error);
    res.status(500).json({ message: 'Failed to create user' });
  }
});

// Settings routes
app.get('/api/settings/:key', async (req, res) => {
  try {
    const { key } = req.params;
    const setting = await storage.getSetting(key);
    if (!setting) {
      return res.status(404).json({ message: 'Setting not found' });
    }
    res.json(setting);
  } catch (error) {
    console.error('Get setting error:', error);
    res.status(500).json({ message: 'Failed to fetch setting' });
  }
});

app.put('/api/settings/:key', authenticateToken, async (req, res) => {
  try {
    const { key } = req.params;
    const { value } = req.body;
    
    const existingSetting = await storage.getSetting(key);
    if (existingSetting) {
      const setting = await storage.updateSetting(key, value);
      res.json(setting);
    } else {
      const setting = await storage.setSetting({ key, value });
      res.status(201).json(setting);
    }
  } catch (error) {
    console.error('Update setting error:', error);
    res.status(500).json({ message: 'Failed to update setting' });
  }
});

// Dashboard stats
app.get('/api/stats', authenticateToken, async (req, res) => {
  try {
    const [menuItems, categories, feedbackList, users] = await Promise.all([
      storage.getAllMenuItems(),
      storage.getAllCategories(),
      storage.getAllFeedback(),
      storage.getAllUsers()
    ]);

    const stats = {
      totalItems: menuItems.length,
      activeCategories: categories.filter(c => c.is_active).length,
      feedbackCount: feedbackList.length,
      activeUsers: users.filter(u => u.is_active).length
    };

    res.json(stats);
  } catch (error) {
    console.error('Get stats error:', error);
    res.status(500).json({ message: 'Failed to fetch stats' });
  }
});

// Serve React app for all non-API routes
app.get('*', (req, res) => {
  res.sendFile(path.join(__dirname, 'dist', 'index.html'));
});

// Error handling middleware
app.use((err, req, res, next) => {
  console.error('Server error:', err);
  const status = err.status || err.statusCode || 500;
  const message = err.message || 'Internal Server Error';
  res.status(status).json({ message });
});

// Start server
const port = process.env.PORT || 5000;
app.listen(port, '0.0.0.0', () => {
  console.log(`Menu Management System running on port ${port}`);
  console.log(`Customer Menu: http://localhost:${port}`);
  console.log(`Admin Panel: http://localhost:${port}/admin`);
  console.log(`Default login: admin / admin123`);
});