import { createContext, useContext, useState, useEffect, ReactNode } from "react";

type Language = 'en' | 'ar';

interface LanguageContextType {
  language: Language;
  toggleLanguage: () => void;
  t: (key: string, arabicText?: string) => string;
}

const LanguageContext = createContext<LanguageContextType | undefined>(undefined);

interface LanguageProviderProps {
  children: ReactNode;
}

export function LanguageProvider({ children }: LanguageProviderProps) {
  const [language, setLanguage] = useState<Language>('en');

  const toggleLanguage = () => {
    const newLanguage = language === 'en' ? 'ar' : 'en';
    setLanguage(newLanguage);
    
    // Update document attributes for RTL support
    const html = document.documentElement;
    const body = document.body;
    
    if (newLanguage === 'ar') {
      html.setAttribute('dir', 'rtl');
      html.setAttribute('lang', 'ar');
      body.classList.add('rtl');
    } else {
      html.setAttribute('dir', 'ltr');
      html.setAttribute('lang', 'en');
      body.classList.remove('rtl');
    }
  };

  // Simple translation function - in a real app, this would use proper i18n
  const t = (key: string, arabicText?: string) => {
    if (language === 'ar' && arabicText) {
      return arabicText;
    }
    return key;
  };

  useEffect(() => {
    // Set initial direction
    const html = document.documentElement;
    html.setAttribute('dir', 'ltr');
    html.setAttribute('lang', 'en');
  }, []);

  return (
    <LanguageContext.Provider value={{ language, toggleLanguage, t }}>
      {children}
    </LanguageContext.Provider>
  );
}

export function useLanguage() {
  const context = useContext(LanguageContext);
  if (context === undefined) {
    throw new Error('useLanguage must be used within a LanguageProvider');
  }
  return context;
}
