import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Skeleton } from "@/components/ui/skeleton";
import { useLanguage } from "@/hooks/use-language";
import { ItemModal } from "./item-modal";
import { Play, Clock } from "lucide-react";
import type { Category, MenuItem } from "@shared/schema";

export function MenuGrid() {
  const { language, t } = useLanguage();
  const [selectedCategory, setSelectedCategory] = useState<number | null>(null);
  const [selectedItem, setSelectedItem] = useState<MenuItem | null>(null);

  const { data: categories, isLoading: categoriesLoading } = useQuery<Category[]>({
    queryKey: ['/api/categories'],
  });

  const { data: menuItems, isLoading: categoryItemsLoading } = useQuery<MenuItem[]>({
    queryKey: ['/api/menu-items', selectedCategory],
    enabled: selectedCategory !== null,
  });

  const { data: allItems, isLoading: allItemsLoading } = useQuery<MenuItem[]>({
    queryKey: ['/api/menu-items'],
    enabled: selectedCategory === null,
  });

  const displayItems = selectedCategory ? menuItems : allItems;
  const itemsLoading = selectedCategory ? categoryItemsLoading : allItemsLoading;

  const formatPrice = (price: string) => {
    return `${parseFloat(price).toFixed(3)} KWD`;
  };

  if (categoriesLoading) {
    return (
      <div className="space-y-6">
        <div className="flex overflow-x-auto space-x-4 pb-4">
          {[...Array(5)].map((_, i) => (
            <Skeleton key={i} className="h-10 w-24 rounded-full flex-shrink-0" />
          ))}
        </div>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {[...Array(6)].map((_, i) => (
            <Card key={i} className="overflow-hidden">
              <Skeleton className="h-48 w-full" />
              <CardContent className="p-4">
                <Skeleton className="h-6 w-3/4 mb-2" />
                <Skeleton className="h-4 w-full mb-3" />
                <Skeleton className="h-10 w-full" />
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-8">
      {/* Category Navigation */}
      <div className="bg-white border-b sticky top-0 z-40">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex overflow-x-auto scrollbar-hide py-4 space-x-6">
            <Button
              variant={selectedCategory === null ? "default" : "outline"}
              className="flex-shrink-0 rounded-full"
              onClick={() => setSelectedCategory(null)}
            >
              {t('All Items', 'جميع العناصر')}
            </Button>
            {categories?.map((category) => (
              <Button
                key={category.id}
                variant={selectedCategory === category.id ? "default" : "outline"}
                className="flex-shrink-0 rounded-full"
                onClick={() => setSelectedCategory(category.id)}
              >
                {language === 'ar' ? category.nameArabic : category.name}
              </Button>
            ))}
          </div>
        </div>
      </div>

      {/* Menu Items Grid */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {selectedCategory && categories && (
          <div className="mb-8">
            {(() => {
              const category = categories.find(c => c.id === selectedCategory);
              if (!category) return null;
              
              return (
                <div>
                  <h2 className="text-3xl font-bold text-gray-900 mb-2">
                    {language === 'ar' ? category.nameArabic : category.name}
                  </h2>
                  {(category.description || category.descriptionArabic) && (
                    <p className="text-gray-600 mb-2">
                      {language === 'ar' ? category.descriptionArabic : category.description}
                    </p>
                  )}
                  {(category.startTime && category.endTime) && (
                    <div className="text-sm text-gray-500 flex items-center">
                      <Clock className="h-4 w-4 mr-1" />
                      <span>
                        {t('Available', 'متاح')}: {category.startTime} - {category.endTime}
                      </span>
                    </div>
                  )}
                </div>
              );
            })()}
          </div>
        )}

        {itemsLoading ? (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {[...Array(6)].map((_, i) => (
              <Card key={i} className="overflow-hidden">
                <Skeleton className="h-48 w-full" />
                <CardContent className="p-4">
                  <Skeleton className="h-6 w-3/4 mb-2" />
                  <Skeleton className="h-4 w-full mb-3" />
                  <Skeleton className="h-10 w-full" />
                </CardContent>
              </Card>
            ))}
          </div>
        ) : displayItems && displayItems.length > 0 ? (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {displayItems.map((item) => (
              <Card
                key={item.id}
                className="overflow-hidden hover:shadow-lg transition-shadow cursor-pointer"
                onClick={() => setSelectedItem(item)}
              >
                <div className="relative h-48">
                  {item.image ? (
                    <img
                      src={item.image}
                      alt={language === 'ar' ? item.nameArabic : item.name}
                      className="w-full h-full object-cover"
                    />
                  ) : (
                    <div className="w-full h-full bg-gray-200 flex items-center justify-center">
                      <span className="text-gray-400">
                        {t('No Image', 'لا توجد صورة')}
                      </span>
                    </div>
                  )}
                  
                  {item.video && (
                    <div className="absolute inset-0 bg-black bg-opacity-40 flex items-center justify-center opacity-0 hover:opacity-100 transition-opacity">
                      <Button size="sm" className="rounded-full">
                        <Play className="h-4 w-4" />
                      </Button>
                    </div>
                  )}
                  
                  <div className="absolute top-4 right-4">
                    <Badge variant="secondary" className="bg-orange-600 text-white">
                      {formatPrice(item.price)}
                    </Badge>
                  </div>
                </div>
                
                <CardContent className="p-4">
                  <h4 className="text-lg font-semibold text-gray-900 mb-2">
                    {language === 'ar' ? item.nameArabic : item.name}
                  </h4>
                  <p className="text-gray-600 text-sm mb-3 line-clamp-2">
                    {language === 'ar' ? item.descriptionArabic : item.description}
                  </p>
                  
                  <Button className="w-full bg-orange-600 hover:bg-orange-700">
                    {t('View Details', 'عرض التفاصيل')}
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        ) : (
          <div className="text-center py-12">
            <p className="text-gray-500 text-lg">
              {t('No menu items available', 'لا توجد عناصر في القائمة')}
            </p>
          </div>
        )}
      </div>

      {/* Item Detail Modal */}
      {selectedItem && (
        <ItemModal
          item={selectedItem}
          isOpen={!!selectedItem}
          onClose={() => setSelectedItem(null)}
        />
      )}
    </div>
  );
}
