import { useState } from "react";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Checkbox } from "@/components/ui/checkbox";
import { Badge } from "@/components/ui/badge";
import { Heart, Play } from "lucide-react";
import { useLanguage } from "@/hooks/use-language";
import type { MenuItem } from "@shared/schema";

interface ItemModalProps {
  item: MenuItem;
  isOpen: boolean;
  onClose: () => void;
}

export function ItemModal({ item, isOpen, onClose }: ItemModalProps) {
  const { language, t } = useLanguage();
  const [selectedModifiers, setSelectedModifiers] = useState<Set<number>>(new Set());
  const [selectedAddons, setSelectedAddons] = useState<Set<number>>(new Set());

  const formatPrice = (price: string) => {
    return `${parseFloat(price).toFixed(3)} KWD`;
  };

  const basePrice = parseFloat(item.price);
  const total = basePrice; // In a real app, add modifier/addon prices

  const handleModifierChange = (modifierId: number, checked: boolean) => {
    const newSelected = new Set(selectedModifiers);
    if (checked) {
      newSelected.add(modifierId);
    } else {
      newSelected.delete(modifierId);
    }
    setSelectedModifiers(newSelected);
  };

  const handleAddonChange = (addonId: number, checked: boolean) => {
    const newSelected = new Set(selectedAddons);
    if (checked) {
      newSelected.add(addonId);
    } else {
      newSelected.delete(addonId);
    }
    setSelectedAddons(newSelected);
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl max-h-screen overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="text-2xl font-bold">
            {language === 'ar' ? item.nameArabic : item.name}
          </DialogTitle>
        </DialogHeader>
        
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          <div>
            {item.image ? (
              <img
                src={item.image}
                alt={language === 'ar' ? item.nameArabic : item.name}
                className="w-full h-80 object-cover rounded-lg"
              />
            ) : (
              <div className="w-full h-80 bg-gray-200 rounded-lg flex items-center justify-center">
                <span className="text-gray-400">
                  {t('No Image', 'لا توجد صورة')}
                </span>
              </div>
            )}
            
            {item.video && (
              <div className="mt-4">
                <video
                  controls
                  className="w-full h-60 rounded-lg"
                  poster={item.image || undefined}
                >
                  <source src={item.video} type="video/mp4" />
                  {t('Your browser does not support the video tag.', 'متصفحك لا يدعم عرض الفيديو.')}
                </video>
              </div>
            )}
          </div>
          
          <div>
            <div className="mb-6">
              <Badge variant="secondary" className="text-3xl font-bold text-orange-600 mb-2">
                {formatPrice(item.price)}
              </Badge>
              <p className="text-gray-600 text-lg">
                {language === 'ar' ? item.descriptionArabic : item.description}
              </p>
            </div>
            
            {/* Mock Modifiers Section - In real app, fetch from API */}
            <div className="mb-6">
              <h3 className="text-lg font-semibold text-gray-800 mb-3">
                {t('Choose Your Extras', 'اختر الإضافات')}
              </h3>
              <div className="space-y-2">
                {/* Mock modifiers - replace with real data */}
                <label className="flex items-center justify-between p-3 border rounded-lg hover:bg-gray-50 cursor-pointer">
                  <div className="flex items-center">
                    <Checkbox
                      checked={selectedModifiers.has(1)}
                      onCheckedChange={(checked) => handleModifierChange(1, checked as boolean)}
                      className="mr-3"
                    />
                    <span className="text-gray-700">
                      {t('Extra Syrup', 'شراب إضافي')}
                    </span>
                  </div>
                  <span className="text-orange-600 font-medium">+0.500 KWD</span>
                </label>
                
                <label className="flex items-center justify-between p-3 border rounded-lg hover:bg-gray-50 cursor-pointer">
                  <div className="flex items-center">
                    <Checkbox
                      checked={selectedModifiers.has(2)}
                      onCheckedChange={(checked) => handleModifierChange(2, checked as boolean)}
                      className="mr-3"
                    />
                    <span className="text-gray-700">
                      {t('Fresh Berries', 'توت طازج')}
                    </span>
                  </div>
                  <span className="text-orange-600 font-medium">+1.250 KWD</span>
                </label>
              </div>
            </div>
            
            {/* Mock Addons Section */}
            <div className="mb-6">
              <h3 className="text-lg font-semibold text-gray-800 mb-3">
                {t('Add-ons', 'الإضافات')}
              </h3>
              <div className="space-y-2">
                <label className="flex items-center justify-between p-3 border rounded-lg hover:bg-gray-50 cursor-pointer">
                  <div className="flex items-center">
                    <Checkbox
                      checked={selectedAddons.has(1)}
                      onCheckedChange={(checked) => handleAddonChange(1, checked as boolean)}
                      className="mr-3"
                    />
                    <span className="text-gray-700">
                      {t('Orange Juice', 'عصير برتقال')}
                    </span>
                  </div>
                  <span className="text-orange-600 font-medium">+2.500 KWD</span>
                </label>
                
                <label className="flex items-center justify-between p-3 border rounded-lg hover:bg-gray-50 cursor-pointer">
                  <div className="flex items-center">
                    <Checkbox
                      checked={selectedAddons.has(2)}
                      onCheckedChange={(checked) => handleAddonChange(2, checked as boolean)}
                      className="mr-3"
                    />
                    <span className="text-gray-700">
                      {t('Coffee', 'قهوة')}
                    </span>
                  </div>
                  <span className="text-orange-600 font-medium">+1.500 KWD</span>
                </label>
              </div>
            </div>
            
            <div className="border-t pt-4">
              <div className="text-xl font-bold text-gray-900 mb-4">
                {t('Total', 'المجموع')}: {formatPrice(total.toString())}
              </div>
              <Button className="w-full bg-orange-600 hover:bg-orange-700">
                <Heart className="h-4 w-4 mr-2" />
                {t('Add to Favorites', 'إضافة للمفضلة')}
              </Button>
            </div>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}
