import { useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { useMutation } from "@tanstack/react-query";
import { z } from "zod";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Checkbox } from "@/components/ui/checkbox";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { useToast } from "@/hooks/use-toast";
import { useLanguage } from "@/hooks/use-language";
import { apiRequest } from "@/lib/queryClient";
import { Send } from "lucide-react";

const feedbackSchema = z.object({
  name: z.string().min(2, "Name must be at least 2 characters"),
  phone: z.string().min(8, "Please enter a valid phone number"),
  email: z.string().email("Please enter a valid email").optional().or(z.literal("")),
  message: z.string().min(10, "Message must be at least 10 characters"),
  consent: z.boolean().refine(val => val === true, "You must agree to the privacy policy"),
});

type FeedbackForm = z.infer<typeof feedbackSchema>;

export function FeedbackForm() {
  const { language, t } = useLanguage();
  const { toast } = useToast();

  const form = useForm<FeedbackForm>({
    resolver: zodResolver(feedbackSchema),
    defaultValues: {
      name: "",
      phone: "",
      email: "",
      message: "",
      consent: false,
    },
  });

  const feedbackMutation = useMutation({
    mutationFn: async (data: FeedbackForm) => {
      const { consent, ...feedbackData } = data;
      return await apiRequest('POST', '/api/feedback', {
        ...feedbackData,
        email: feedbackData.email || null,
      });
    },
    onSuccess: () => {
      toast({
        title: t("Thank you!", "شكراً لك!"),
        description: t("Your feedback has been submitted successfully.", "تم إرسال ملاحظاتك بنجاح."),
      });
      form.reset();
    },
    onError: (error) => {
      toast({
        title: t("Error", "خطأ"),
        description: t("Failed to submit feedback. Please try again.", "فشل في إرسال الملاحظات. يرجى المحاولة مرة أخرى."),
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: FeedbackForm) => {
    feedbackMutation.mutate(data);
  };

  return (
    <section className="bg-gray-800 py-16 mt-16">
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-8">
          <h2 className="text-3xl font-bold text-white mb-4">
            {t("We Value Your Feedback", "نحن نقدر ملاحظاتك")}
          </h2>
          <p className="text-gray-300">
            {t("Help us improve by sharing your experience", "ساعدنا في التحسين من خلال مشاركة تجربتك")}
          </p>
        </div>
        
        <div className="bg-white rounded-xl p-8 shadow-lg">
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <FormField
                  control={form.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>
                        {t("Name", "الاسم")} <span className="text-red-500">*</span>
                      </FormLabel>
                      <FormControl>
                        <Input
                          {...field}
                          placeholder={t("Enter your full name", "أدخل اسمك الكامل")}
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                
                <FormField
                  control={form.control}
                  name="phone"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>
                        {t("Phone Number", "رقم الهاتف")} <span className="text-red-500">*</span>
                      </FormLabel>
                      <FormControl>
                        <Input
                          {...field}
                          type="tel"
                          placeholder={t("+965 XXXX XXXX", "+965 XXXX XXXX")}
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>
              
              <FormField
                control={form.control}
                name="email"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>
                      {t("Email Address", "البريد الإلكتروني")} <span className="text-gray-500">({t("Optional", "اختياري")})</span>
                    </FormLabel>
                    <FormControl>
                      <Input
                        {...field}
                        type="email"
                        placeholder={t("your.email@example.com", "your.email@example.com")}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              
              <FormField
                control={form.control}
                name="message"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>
                      {t("Your Feedback", "ملاحظاتك")} <span className="text-red-500">*</span>
                    </FormLabel>
                    <FormControl>
                      <Textarea
                        {...field}
                        rows={4}
                        placeholder={t("Share your thoughts about our menu and service...", "شاركنا آراءك حول قائمتنا وخدمتنا...")}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              
              <div className="flex items-center justify-between">
                <FormField
                  control={form.control}
                  name="consent"
                  render={({ field }) => (
                    <FormItem className="flex flex-row items-start space-x-3 space-y-0">
                      <FormControl>
                        <Checkbox
                          checked={field.value}
                          onCheckedChange={field.onChange}
                        />
                      </FormControl>
                      <div className="space-y-1 leading-none">
                        <FormLabel className="text-sm text-gray-600">
                          {t("I agree to the privacy policy and terms of service", "أوافق على سياسة الخصوصية وشروط الخدمة")}
                        </FormLabel>
                        <FormMessage />
                      </div>
                    </FormItem>
                  )}
                />
                
                <Button
                  type="submit"
                  disabled={feedbackMutation.isPending}
                  className="bg-orange-600 hover:bg-orange-700"
                >
                  <Send className="h-4 w-4 mr-2" />
                  {feedbackMutation.isPending
                    ? t("Submitting...", "جاري الإرسال...")
                    : t("Submit Feedback", "إرسال الملاحظات")
                  }
                </Button>
              </div>
            </form>
          </Form>
        </div>
      </div>
    </section>
  );
}
