import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { setAuthHeader } from "@/lib/auth-utils";
import { Grid3X3, List, Layout, Square, Image, IndentDecrease } from "lucide-react";

interface Template {
  id: string;
  name: string;
  description: string;
  icon: string;
  preview: string;
}

const availableTemplates: Template[] = [
  {
    id: "classic-grid",
    name: "Classic Grid",
    description: "Traditional grid layout with equal-sized cards",
    icon: "grid",
    preview: "from-blue-100 to-blue-200",
  },
  {
    id: "modern-list",
    name: "Modern List",
    description: "Clean list view with horizontal layout",
    icon: "list",
    preview: "from-green-100 to-green-200",
  },
  {
    id: "magazine-style",
    name: "Magazine Style",
    description: "Asymmetric layout with featured items",
    icon: "layout",
    preview: "from-purple-100 to-purple-200",
  },
  {
    id: "minimal-cards",
    name: "Minimal Cards",
    description: "Clean and simple card-based design",
    icon: "square",
    preview: "from-orange-100 to-orange-200",
  },
  {
    id: "hero-style",
    name: "Full-Width Hero",
    description: "Large featured items with smaller grid",
    icon: "image",
    preview: "from-red-100 to-red-200",
  },
  {
    id: "category-tabs",
    name: "Category IndentDecrease",
    description: "Tabbed interface for easy navigation",
    icon: "tabs",
    preview: "from-indigo-100 to-indigo-200",
  },
];

export function TemplateManagement() {
  const [selectedTemplate, setSelectedTemplate] = useState<string>('classic-grid');
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data: currentTemplate } = useQuery({
    queryKey: ['/api/settings/current_template'],
    queryFn: async () => {
      const response = await fetch('/api/settings/current_template', {
        headers: setAuthHeader(),
      });
      if (response.status === 404) {
        return { key: 'current_template', value: 'classic-grid' };
      }
      if (!response.ok) throw new Error('Failed to fetch template setting');
      return response.json();
    },
  });

  const updateTemplateMutation = useMutation({
    mutationFn: async (templateId: string) => {
      const response = await fetch('/api/settings/current_template', {
        method: 'PUT',
        headers: {
          ...setAuthHeader(),
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ value: templateId }),
      });
      if (!response.ok) throw new Error('Failed to update template');
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/settings/current_template'] });
      toast({
        title: "Success",
        description: "Template updated successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: "Failed to update template",
        variant: "destructive",
      });
    },
  });

  const handleSelectTemplate = (templateId: string) => {
    updateTemplateMutation.mutate(templateId);
  };

  const getTemplateIcon = (iconName: string) => {
    switch (iconName) {
      case 'grid':
        return <Grid3X3 className="h-6 w-6" />;
      case 'list':
        return <List className="h-6 w-6" />;
      case 'layout':
        return <Layout className="h-6 w-6" />;
      case 'square':
        return <Square className="h-6 w-6" />;
      case 'image':
        return <Image className="h-6 w-6" />;
      case 'tabs':
        return <IndentDecrease className="h-6 w-6" />;
      default:
        return <Grid3X3 className="h-6 w-6" />;
    }
  };

  const getCurrentTemplate = () => {
    const templateId = currentTemplate?.value || 'classic-grid';
    return availableTemplates.find(t => t.id === templateId) || availableTemplates[0];
  };

  return (
    <div className="p-6">
      <div className="mb-8">
        <h2 className="text-3xl font-bold text-gray-900 mb-2">Menu Templates</h2>
        <p className="text-gray-600">Choose from various layout templates for your menu display</p>
      </div>
      
      {/* Template Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-8">
        {availableTemplates.map((template) => (
          <Card
            key={template.id}
            className={`overflow-hidden hover:shadow-lg transition-all cursor-pointer border-2 ${
              currentTemplate?.value === template.id 
                ? 'border-orange-500 shadow-lg' 
                : 'border-gray-200 hover:border-gray-300'
            }`}
          >
            <div className={`aspect-video bg-gradient-to-br ${template.preview} p-4`}>
              <div className="h-full bg-white rounded-lg p-3 shadow-sm">
                {/* Template Preview Mockup */}
                {template.id === 'classic-grid' && (
                  <div className="grid grid-cols-3 gap-2 h-full">
                    {[...Array(6)].map((_, i) => (
                      <div key={i} className="bg-gray-200 rounded"></div>
                    ))}
                  </div>
                )}
                
                {template.id === 'modern-list' && (
                  <div className="space-y-2 h-full">
                    {[...Array(3)].map((_, i) => (
                      <div key={i} className="flex space-x-2">
                        <div className="w-12 bg-gray-200 rounded"></div>
                        <div className="flex-1 bg-gray-200 rounded"></div>
                      </div>
                    ))}
                  </div>
                )}
                
                {template.id === 'magazine-style' && (
                  <div className="grid grid-cols-2 gap-2 h-full">
                    <div className="bg-gray-200 rounded row-span-2"></div>
                    <div className="bg-gray-200 rounded"></div>
                    <div className="bg-gray-200 rounded"></div>
                  </div>
                )}
                
                {template.id === 'minimal-cards' && (
                  <div className="grid grid-cols-2 gap-3 h-full">
                    {[...Array(2)].map((_, i) => (
                      <div key={i} className="space-y-2">
                        <div className="bg-gray-200 rounded h-8"></div>
                        <div className="bg-gray-100 rounded h-4"></div>
                      </div>
                    ))}
                  </div>
                )}
                
                {template.id === 'hero-style' && (
                  <div className="space-y-2 h-full">
                    <div className="bg-gray-200 rounded h-16"></div>
                    <div className="grid grid-cols-3 gap-1 flex-1">
                      {[...Array(3)].map((_, i) => (
                        <div key={i} className="bg-gray-100 rounded"></div>
                      ))}
                    </div>
                  </div>
                )}
                
                {template.id === 'category-tabs' && (
                  <div className="space-y-2 h-full">
                    <div className="flex space-x-1">
                      <div className="bg-orange-600 rounded px-2 py-1 text-xs text-white">Tab 1</div>
                      <div className="bg-gray-200 rounded px-2 py-1 text-xs">Tab 2</div>
                      <div className="bg-gray-200 rounded px-2 py-1 text-xs">Tab 3</div>
                    </div>
                    <div className="grid grid-cols-2 gap-2 flex-1">
                      {[...Array(4)].map((_, i) => (
                        <div key={i} className="bg-gray-200 rounded"></div>
                      ))}
                    </div>
                  </div>
                )}
              </div>
            </div>
            <div className="p-4">
              <h3 className="font-semibold text-lg mb-2">{template.name}</h3>
              <p className="text-gray-600 text-sm mb-3">{template.description}</p>
              <Button
                className={`w-full ${
                  currentTemplate?.value === template.id
                    ? 'bg-orange-600 hover:bg-orange-700'
                    : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                }`}
                onClick={() => handleSelectTemplate(template.id)}
                disabled={updateTemplateMutation.isPending}
              >
                {currentTemplate?.value === template.id ? 'Selected' : 'Select Template'}
              </Button>
            </div>
          </Card>
        ))}
      </div>
      
      {/* Current Template Status */}
      <Card>
        <CardHeader>
          <CardTitle>Current Template</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <div className="w-16 h-16 bg-orange-100 rounded-lg flex items-center justify-center">
                {getTemplateIcon(getCurrentTemplate().icon)}
              </div>
              <div>
                <h4 className="font-medium text-gray-900">{getCurrentTemplate().name}</h4>
                <p className="text-sm text-gray-600">{getCurrentTemplate().description}</p>
              </div>
            </div>
            <div className="text-right">
              <Badge className="bg-green-100 text-green-800">Active</Badge>
              <p className="text-xs text-gray-500 mt-1">
                Updated: {new Date().toLocaleDateString()}
              </p>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
