import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { setAuthHeader } from "@/lib/auth-utils";
import { apiRequest } from "@/lib/queryClient";
import { MessageCircle, Calendar, Star, Download, Reply, Archive } from "lucide-react";
import type { Feedback } from "@shared/schema";

export function FeedbackManagement() {
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data: feedbackList, isLoading } = useQuery<Feedback[]>({
    queryKey: ['/api/feedback'],
    queryFn: async () => {
      const response = await fetch('/api/feedback', {
        headers: setAuthHeader(),
      });
      if (!response.ok) throw new Error('Failed to fetch feedback');
      return response.json();
    },
  });

  const archiveFeedbackMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest('PUT', `/api/feedback/${id}/archive`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/feedback'] });
      toast({
        title: "Success",
        description: "Feedback archived successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: "Failed to archive feedback",
        variant: "destructive",
      });
    },
  });

  const handleArchive = (id: number) => {
    archiveFeedbackMutation.mutate(id);
  };

  const handleExport = () => {
    // In a real app, this would generate and download a CSV/Excel file
    toast({
      title: "Export",
      description: "Feedback export feature will be implemented",
    });
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
    });
  };

  const getInitials = (name: string) => {
    return name
      .split(' ')
      .map(word => word.charAt(0))
      .join('')
      .toUpperCase()
      .slice(0, 2);
  };

  const renderStars = (rating: number) => {
    return [...Array(5)].map((_, i) => (
      <Star
        key={i}
        className={`h-4 w-4 ${
          i < rating ? 'text-yellow-400 fill-current' : 'text-gray-300'
        }`}
      />
    ));
  };

  if (isLoading) {
    return (
      <div className="p-6">
        <div className="mb-8">
          <Skeleton className="h-8 w-48 mb-2" />
          <Skeleton className="h-4 w-96" />
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
          {[...Array(3)].map((_, i) => (
            <Card key={i}>
              <CardContent className="p-6">
                <Skeleton className="h-12 w-12 rounded-full mb-4" />
                <Skeleton className="h-4 w-24 mb-2" />
                <Skeleton className="h-8 w-16" />
              </CardContent>
            </Card>
          ))}
        </div>
        
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-32" />
          </CardHeader>
          <CardContent>
            <div className="space-y-6">
              {[...Array(3)].map((_, i) => (
                <div key={i} className="p-6 border-b">
                  <div className="flex items-start justify-between mb-4">
                    <div className="flex items-center space-x-3">
                      <Skeleton className="h-10 w-10 rounded-full" />
                      <div>
                        <Skeleton className="h-4 w-32 mb-1" />
                        <Skeleton className="h-3 w-48" />
                      </div>
                    </div>
                    <Skeleton className="h-6 w-20" />
                  </div>
                  <Skeleton className="h-16 w-full mb-4" />
                  <div className="flex justify-between">
                    <div className="flex space-x-2">
                      <Skeleton className="h-6 w-16" />
                      <Skeleton className="h-6 w-20" />
                    </div>
                    <div className="flex space-x-2">
                      <Skeleton className="h-8 w-16" />
                      <Skeleton className="h-8 w-16" />
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  const stats = {
    total: feedbackList?.length || 0,
    thisMonth: feedbackList?.filter(f => {
      const feedbackDate = new Date(f.createdAt || '');
      const now = new Date();
      return feedbackDate.getMonth() === now.getMonth() && 
             feedbackDate.getFullYear() === now.getFullYear();
    }).length || 0,
    avgRating: feedbackList?.length 
      ? (feedbackList.reduce((sum, f) => sum + (f.rating || 5), 0) / feedbackList.length).toFixed(1)
      : '0.0',
  };

  return (
    <div className="p-6">
      <div className="mb-8">
        <h2 className="text-3xl font-bold text-gray-900 mb-2">Customer Feedback</h2>
        <p className="text-gray-600">Review and manage customer feedback submissions</p>
      </div>
      
      {/* Feedback Stats */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-gray-600 text-sm">Total Feedback</p>
                <p className="text-3xl font-bold text-gray-900">{stats.total}</p>
              </div>
              <div className="bg-blue-100 p-3 rounded-full">
                <MessageCircle className="text-blue-600 h-6 w-6" />
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-gray-600 text-sm">This Month</p>
                <p className="text-3xl font-bold text-gray-900">{stats.thisMonth}</p>
              </div>
              <div className="bg-green-100 p-3 rounded-full">
                <Calendar className="text-green-600 h-6 w-6" />
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-gray-600 text-sm">Avg Rating</p>
                <p className="text-3xl font-bold text-gray-900">{stats.avgRating}</p>
              </div>
              <div className="bg-yellow-100 p-3 rounded-full">
                <Star className="text-yellow-600 h-6 w-6" />
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
      
      {/* Feedback List */}
      <Card>
        <div className="p-6 border-b border-gray-200">
          <div className="flex items-center justify-between">
            <h3 className="text-lg font-semibold text-gray-900">Recent Feedback</h3>
            <div className="flex space-x-3">
              <Select>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="All Feedback" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Feedback</SelectItem>
                  <SelectItem value="week">This Week</SelectItem>
                  <SelectItem value="month">This Month</SelectItem>
                </SelectContent>
              </Select>
              <Button onClick={handleExport} className="bg-orange-600 hover:bg-orange-700">
                <Download className="h-4 w-4 mr-2" />
                Export
              </Button>
            </div>
          </div>
        </div>
        
        <CardContent className="p-0">
          {feedbackList && feedbackList.length > 0 ? (
            <div className="divide-y divide-gray-200">
              {feedbackList.map((feedback) => (
                <div key={feedback.id} className="p-6 hover:bg-gray-50 transition-colors">
                  <div className="flex items-start justify-between mb-4">
                    <div className="flex items-center space-x-3">
                      <div className="w-10 h-10 bg-orange-100 rounded-full flex items-center justify-center">
                        <span className="text-orange-600 font-semibold text-sm">
                          {getInitials(feedback.name)}
                        </span>
                      </div>
                      <div>
                        <h4 className="font-medium text-gray-900">{feedback.name}</h4>
                        <div className="flex items-center space-x-2 text-sm text-gray-500">
                          <span>{feedback.phone}</span>
                          {feedback.email && (
                            <>
                              <span>•</span>
                              <span>{feedback.email}</span>
                            </>
                          )}
                        </div>
                      </div>
                    </div>
                    <div className="text-right">
                      <div className="flex items-center space-x-1 mb-1">
                        {renderStars(feedback.rating || 5)}
                        <span className="text-sm text-gray-600 ml-1">
                          {feedback.rating || 5}.0
                        </span>
                      </div>
                      <p className="text-xs text-gray-500">
                        {formatDate(feedback.createdAt || '')}
                      </p>
                    </div>
                  </div>
                  
                  <p className="text-gray-700 mb-4">{feedback.message}</p>
                  
                  <div className="flex items-center justify-between">
                    <div className="flex space-x-2">
                      <Badge className="bg-green-100 text-green-800">Positive</Badge>
                      <Badge className="bg-blue-100 text-blue-800">Menu Quality</Badge>
                    </div>
                    
                    <div className="flex space-x-2">
                      <Button
                        size="sm"
                        variant="ghost"
                        className="text-orange-600 hover:text-orange-700"
                      >
                        <Reply className="h-4 w-4 mr-1" />
                        Reply
                      </Button>
                      <Button
                        size="sm"
                        variant="ghost"
                        onClick={() => handleArchive(feedback.id)}
                        className="text-gray-600 hover:text-gray-800"
                      >
                        <Archive className="h-4 w-4 mr-1" />
                        Archive
                      </Button>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          ) : (
            <div className="p-12 text-center">
              <MessageCircle className="h-12 w-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">No feedback yet</h3>
              <p className="text-gray-500">Customer feedback will appear here when submitted.</p>
            </div>
          )}
          
          {feedbackList && feedbackList.length > 0 && (
            <div className="p-6 border-t border-gray-200 text-center">
              <Button variant="ghost" className="text-orange-600 hover:text-orange-700">
                Load More Feedback
              </Button>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
