import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Skeleton } from "@/components/ui/skeleton";
import { Button } from "@/components/ui/button";
import { Utensils, Tags, MessageCircle, Users, Plus, FolderPlus, MessageSquare } from "lucide-react";

interface Stats {
  totalItems: number;
  activeCategories: number;
  feedbackCount: number;
  activeUsers: number;
}

export function Dashboard() {
  const { data: stats, isLoading } = useQuery<Stats>({
    queryKey: ['/api/stats'],
  });

  if (isLoading) {
    return (
      <div className="p-6">
        <div className="mb-8">
          <Skeleton className="h-8 w-48 mb-2" />
          <Skeleton className="h-4 w-96" />
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          {[...Array(4)].map((_, i) => (
            <Card key={i}>
              <CardContent className="p-6">
                <Skeleton className="h-12 w-12 rounded-full mb-4" />
                <Skeleton className="h-4 w-24 mb-2" />
                <Skeleton className="h-8 w-16" />
              </CardContent>
            </Card>
          ))}
        </div>
        
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          <Card>
            <CardHeader>
              <Skeleton className="h-6 w-32" />
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {[...Array(3)].map((_, i) => (
                  <Skeleton key={i} className="h-16 w-full" />
                ))}
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardHeader>
              <Skeleton className="h-6 w-32" />
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {[...Array(3)].map((_, i) => (
                  <div key={i} className="flex items-center space-x-3">
                    <Skeleton className="h-2 w-2 rounded-full" />
                    <div className="flex-1">
                      <Skeleton className="h-4 w-full mb-1" />
                      <Skeleton className="h-3 w-20" />
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      <div className="mb-8">
        <h2 className="text-3xl font-bold text-gray-900 mb-2">Dashboard</h2>
        <p className="text-gray-600">Overview of your menu management system</p>
      </div>
      
      {/* Stats Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-gray-600 text-sm">Total Menu Items</p>
                <p className="text-3xl font-bold text-gray-900">{stats?.totalItems || 0}</p>
              </div>
              <div className="bg-orange-100 p-3 rounded-full">
                <Utensils className="text-orange-600 h-6 w-6" />
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-gray-600 text-sm">Active Categories</p>
                <p className="text-3xl font-bold text-gray-900">{stats?.activeCategories || 0}</p>
              </div>
              <div className="bg-green-100 p-3 rounded-full">
                <Tags className="text-green-600 h-6 w-6" />
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-gray-600 text-sm">Customer Feedback</p>
                <p className="text-3xl font-bold text-gray-900">{stats?.feedbackCount || 0}</p>
              </div>
              <div className="bg-blue-100 p-3 rounded-full">
                <MessageCircle className="text-blue-600 h-6 w-6" />
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-gray-600 text-sm">Active Users</p>
                <p className="text-3xl font-bold text-gray-900">{stats?.activeUsers || 0}</p>
              </div>
              <div className="bg-purple-100 p-3 rounded-full">
                <Users className="text-purple-600 h-6 w-6" />
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
      
      {/* Quick Actions and Recent Activity */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
        <Card>
          <CardHeader>
            <CardTitle>Quick Actions</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              <Button
                variant="ghost"
                className="w-full justify-between p-4 h-auto"
                onClick={() => {
                  // Navigate to add item - in a real app, use router
                  console.log('Navigate to add item');
                }}
              >
                <div className="flex items-center">
                  <Plus className="text-orange-600 h-5 w-5 mr-3" />
                  <span className="font-medium">Add New Menu Item</span>
                </div>
                <span className="text-gray-400">→</span>
              </Button>
              
              <Button
                variant="ghost"
                className="w-full justify-between p-4 h-auto"
                onClick={() => {
                  console.log('Navigate to manage categories');
                }}
              >
                <div className="flex items-center">
                  <FolderPlus className="text-green-600 h-5 w-5 mr-3" />
                  <span className="font-medium">Manage Categories</span>
                </div>
                <span className="text-gray-400">→</span>
              </Button>
              
              <Button
                variant="ghost"
                className="w-full justify-between p-4 h-auto"
                onClick={() => {
                  console.log('Navigate to view feedback');
                }}
              >
                <div className="flex items-center">
                  <MessageSquare className="text-blue-600 h-5 w-5 mr-3" />
                  <span className="font-medium">Review Feedback</span>
                </div>
                <span className="text-gray-400">→</span>
              </Button>
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardHeader>
            <CardTitle>Recent Activity</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div className="flex items-center space-x-3">
                <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                <div className="flex-1">
                  <p className="text-sm text-gray-800">System initialized successfully</p>
                  <p className="text-xs text-gray-500">Just now</p>
                </div>
              </div>
              
              <div className="flex items-center space-x-3">
                <div className="w-2 h-2 bg-orange-500 rounded-full"></div>
                <div className="flex-1">
                  <p className="text-sm text-gray-800">Menu management system ready</p>
                  <p className="text-xs text-gray-500">1 minute ago</p>
                </div>
              </div>
              
              <div className="flex items-center space-x-3">
                <div className="w-2 h-2 bg-blue-500 rounded-full"></div>
                <div className="flex-1">
                  <p className="text-sm text-gray-800">Database connection established</p>
                  <p className="text-xs text-gray-500">2 minutes ago</p>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
