# Menu Management System - cPanel Deployment Guide

## System Overview
This is a bilingual (English/Arabic) menu management system built with Node.js, React, and PostgreSQL. It features time-based category availability, admin panel, customer feedback collection, and template management.

## Prerequisites
- cPanel hosting with Node.js support (v18 or higher)
- PostgreSQL database access
- File Manager access in cPanel

## Deployment Steps

### 1. Database Setup

#### Create PostgreSQL Database
1. Log into your cPanel
2. Go to "PostgreSQL Databases"
3. Create a new database (e.g., `menu_system`)
4. Create a database user and assign it to the database
5. Note down the database credentials:
   - Database name
   - Username
   - Password
   - Host (usually localhost)
   - Port (usually 5432)

#### Import Database Schema
Execute the following SQL commands in your PostgreSQL database:

```sql
-- Users table
CREATE TABLE users (
  id SERIAL PRIMARY KEY,
  username TEXT NOT NULL UNIQUE,
  email TEXT NOT NULL UNIQUE,
  password TEXT NOT NULL,
  role TEXT NOT NULL DEFAULT 'admin',
  first_name TEXT,
  last_name TEXT,
  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMP DEFAULT NOW(),
  updated_at TIMESTAMP DEFAULT NOW()
);

-- Categories table
CREATE TABLE categories (
  id SERIAL PRIMARY KEY,
  name TEXT NOT NULL,
  name_arabic TEXT NOT NULL,
  description TEXT,
  description_arabic TEXT,
  start_time TEXT,
  end_time TEXT,
  is_active BOOLEAN DEFAULT true,
  sort_order INTEGER DEFAULT 0,
  created_at TIMESTAMP DEFAULT NOW(),
  updated_at TIMESTAMP DEFAULT NOW()
);

-- Subcategories table
CREATE TABLE subcategories (
  id SERIAL PRIMARY KEY,
  category_id INTEGER REFERENCES categories(id),
  name TEXT NOT NULL,
  name_arabic TEXT NOT NULL,
  description TEXT,
  description_arabic TEXT,
  is_active BOOLEAN DEFAULT true,
  sort_order INTEGER DEFAULT 0,
  created_at TIMESTAMP DEFAULT NOW(),
  updated_at TIMESTAMP DEFAULT NOW()
);

-- Menu items table
CREATE TABLE menu_items (
  id SERIAL PRIMARY KEY,
  category_id INTEGER REFERENCES categories(id),
  subcategory_id INTEGER REFERENCES subcategories(id),
  name TEXT NOT NULL,
  name_arabic TEXT NOT NULL,
  description TEXT,
  description_arabic TEXT,
  price DECIMAL(10,3) NOT NULL,
  image TEXT,
  video TEXT,
  is_active BOOLEAN DEFAULT true,
  sort_order INTEGER DEFAULT 0,
  created_at TIMESTAMP DEFAULT NOW(),
  updated_at TIMESTAMP DEFAULT NOW()
);

-- Modifier categories table
CREATE TABLE modifier_categories (
  id SERIAL PRIMARY KEY,
  name TEXT NOT NULL,
  name_arabic TEXT NOT NULL,
  is_required BOOLEAN DEFAULT false,
  max_selections INTEGER DEFAULT 1,
  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMP DEFAULT NOW(),
  updated_at TIMESTAMP DEFAULT NOW()
);

-- Modifiers table
CREATE TABLE modifiers (
  id SERIAL PRIMARY KEY,
  category_id INTEGER REFERENCES modifier_categories(id),
  name TEXT NOT NULL,
  name_arabic TEXT NOT NULL,
  price DECIMAL(10,3) NOT NULL DEFAULT 0,
  is_active BOOLEAN DEFAULT true,
  sort_order INTEGER DEFAULT 0,
  created_at TIMESTAMP DEFAULT NOW(),
  updated_at TIMESTAMP DEFAULT NOW()
);

-- Menu item modifiers junction table
CREATE TABLE menu_item_modifiers (
  id SERIAL PRIMARY KEY,
  menu_item_id INTEGER REFERENCES menu_items(id),
  modifier_category_id INTEGER REFERENCES modifier_categories(id),
  custom_price DECIMAL(10,3)
);

-- Addon categories table
CREATE TABLE addon_categories (
  id SERIAL PRIMARY KEY,
  name TEXT NOT NULL,
  name_arabic TEXT NOT NULL,
  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMP DEFAULT NOW(),
  updated_at TIMESTAMP DEFAULT NOW()
);

-- Addons table
CREATE TABLE addons (
  id SERIAL PRIMARY KEY,
  category_id INTEGER REFERENCES addon_categories(id),
  name TEXT NOT NULL,
  name_arabic TEXT NOT NULL,
  price DECIMAL(10,3) NOT NULL,
  is_active BOOLEAN DEFAULT true,
  sort_order INTEGER DEFAULT 0,
  created_at TIMESTAMP DEFAULT NOW(),
  updated_at TIMESTAMP DEFAULT NOW()
);

-- Menu item addons junction table
CREATE TABLE menu_item_addons (
  id SERIAL PRIMARY KEY,
  menu_item_id INTEGER REFERENCES menu_items(id),
  addon_category_id INTEGER REFERENCES addon_categories(id)
);

-- Customer feedback table
CREATE TABLE feedback (
  id SERIAL PRIMARY KEY,
  name TEXT NOT NULL,
  phone TEXT NOT NULL,
  email TEXT,
  message TEXT NOT NULL,
  rating INTEGER DEFAULT 5,
  is_archived BOOLEAN DEFAULT false,
  created_at TIMESTAMP DEFAULT NOW(),
  updated_at TIMESTAMP DEFAULT NOW()
);

-- Settings table
CREATE TABLE settings (
  id SERIAL PRIMARY KEY,
  key TEXT NOT NULL UNIQUE,
  value JSONB NOT NULL,
  updated_at TIMESTAMP DEFAULT NOW()
);
```

#### Insert Sample Data (Optional)
```sql
-- Insert default admin user (password: admin123)
INSERT INTO users (username, email, password, role, first_name, last_name) 
VALUES ('admin', 'admin@restaurant.com', '$2b$10$rQJ8YnWKZdOQ0G4Zw7yPzeOcFQ/kGJw8jW3qqQ1Vx3kTbOkzYOF4.', 'super_admin', 'System', 'Administrator');

-- Insert sample categories
INSERT INTO categories (name, name_arabic, description, description_arabic, start_time, end_time, sort_order) VALUES
('Breakfast', 'الإفطار', 'Start your day with our delicious breakfast items', 'ابدأ يومك مع وجبات الإفطار اللذيذة', '06:00', '11:00', 1),
('Lunch', 'الغداء', 'Hearty meals perfect for lunch time', 'وجبات شهية مثالية لوقت الغداء', '11:00', '16:00', 2),
('Dinner', 'العشاء', 'Elegant dinner options for evening dining', 'خيارات عشاء أنيقة لتناول الطعام المسائي', '16:00', '23:00', 3),
('Beverages', 'المشروبات', 'Refreshing drinks available all day', 'مشروبات منعشة متاحة طوال اليوم', '06:00', '23:59', 4),
('Desserts', 'الحلويات', 'Sweet treats to end your meal', 'حلويات لذيذة لإنهاء وجبتك', '12:00', '23:00', 5);

-- Insert default template setting
INSERT INTO settings (key, value) VALUES ('current_template', '"classic-grid"');
```

### 2. File Upload and Setup

#### Upload Files to cPanel
1. Create a new folder in your public_html directory (e.g., `menu-system`)
2. Upload all project files to this folder
3. Extract if uploaded as a zip file

#### Install Dependencies
1. Open Terminal in cPanel (if available) or use SSH
2. Navigate to your project directory
3. Run: `npm install`

#### Environment Configuration
Create a `.env` file in the root directory with your database credentials:

```env
NODE_ENV=production
DATABASE_URL=postgresql://username:password@localhost:5432/database_name
JWT_SECRET=your-secure-jwt-secret-key-here
```

Replace the values with your actual database credentials.

### 3. Build the Application

#### Build Frontend
Run the following command to build the React frontend:
```bash
npm run build
```

#### Start the Application
For production deployment:
```bash
npm start
```

### 4. cPanel Configuration

#### Node.js Setup
1. Go to "Node.js" in cPanel
2. Create a new Node.js app
3. Set the following:
   - Node.js version: 18.x or higher
   - Application mode: Production
   - Application root: `/menu-system` (or your folder name)
   - Application URL: your domain or subdomain
   - Application startup file: `server/index.js`

#### Environment Variables
In the Node.js app settings, add environment variables:
- `NODE_ENV`: production
- `DATABASE_URL`: your PostgreSQL connection string
- `JWT_SECRET`: a secure random string

### 5. Domain Configuration

#### Subdomain Setup (Recommended)
1. Create a subdomain (e.g., menu.yourdomain.com)
2. Point it to your Node.js application
3. The application will be accessible at: https://menu.yourdomain.com

#### Main Domain Setup
If using main domain, configure the Node.js app to serve from root.

### 6. SSL Certificate
1. Go to "SSL/TLS" in cPanel
2. Install SSL certificate for your domain
3. Enable "Force HTTPS Redirect"

## Application Usage

### Admin Access
- URL: `https://yourdomain.com/admin`
- Default Login: `admin` / `admin123`
- Change the default password immediately after first login

### Features Available
1. **Menu Management**: Add/edit categories, subcategories, and menu items
2. **Time-Based Availability**: Categories automatically show/hide based on time
3. **Bilingual Support**: Full English/Arabic with RTL support
4. **Template Management**: Choose from different menu layouts
5. **Customer Feedback**: Collect and manage customer reviews
6. **User Management**: Add admin users with different roles

### Customer Features
- View menu in English or Arabic
- Browse categories (filtered by current time)
- View item details with images/videos
- Submit feedback with ratings

## Troubleshooting

### Common Issues

1. **Database Connection Error**
   - Verify database credentials in .env file
   - Check if PostgreSQL service is running
   - Ensure database exists and user has proper permissions

2. **Application Won't Start**
   - Check Node.js version (minimum 18.x)
   - Verify all dependencies are installed
   - Check error logs in cPanel

3. **Images Not Loading**
   - Ensure image URLs are accessible
   - Check file permissions for uploaded images
   - Verify domain settings

4. **Arabic Text Issues**
   - Ensure UTF-8 encoding is set
   - Check browser language settings
   - Verify Arabic fonts are loading

### Performance Optimization
1. Enable gzip compression in cPanel
2. Use CDN for image assets
3. Enable caching in the application
4. Monitor database performance

## Security Considerations
1. Change default admin password
2. Use strong JWT secret
3. Enable SSL/HTTPS
4. Regular database backups
5. Keep dependencies updated
6. Monitor access logs

## Support and Maintenance
- Regular database backups
- Monitor application logs
- Update menu items through admin panel
- Check feedback submissions regularly
- Update pricing and availability as needed

## File Structure
```
menu-system/
├── server/           # Backend API server
├── client/           # React frontend
├── shared/           # Shared types and schemas
├── package.json      # Dependencies
├── .env             # Environment variables
├── drizzle.config.ts # Database configuration
└── DEPLOYMENT_GUIDE.md # This file
```

## Contact
For technical support or questions about deployment, refer to your hosting provider's documentation or contact their support team.